jQuery(document).ready(function($) {
    // Safe color picker initialization with error handling
    function initializeColorPickers() {
        if (typeof $.fn.wpColorPicker === 'function') {
            try {
                $('.lpg-color-picker').wpColorPicker({
                    defaultColor: false,
                    change: function(event, ui) {
                        updateButtonPreview();
                        updatePaginationPreview();
                    },
                    clear: function() {
                        updateButtonPreview();
                        updatePaginationPreview();
                    },
                    hide: true,
                    palettes: true
                });
                console.log('Color pickers initialized successfully');
            } catch (error) {
                console.error('Error initializing color pickers:', error);
            }
        } else {
            console.warn('WordPress Color Picker not available');
        }
    }
    
    // Initialize color pickers
    initializeColorPickers();
    
    // Initialize tabs with better error handling
    function initializeTabs() {
        try {
            if (typeof $.fn.tabs === 'function') {
                $('#lpg-admin-tabs').tabs({
                    active: 0,
                    activate: function(event, ui) {
                        if (ui.newPanel.attr('id') === 'lpg-tabs-1') {
                            if (!$('#lpg-shortcode-builder').data('editing')) {
                                resetForm();
                            }
                        }
                    }
                });
                console.log('jQuery UI Tabs initialized successfully');
            } else {
                console.log('jQuery UI Tabs not available, using fallback');
                $('.lpg-tab-nav a').on('click', function(e) {
                    e.preventDefault();
                    var target = $(this).attr('href');
                    
                    $('.lpg-tab-content').hide();
                    $('.lpg-tab-nav li').removeClass('ui-tabs-active');
                    
                    $(target).show();
                    $(this).parent().addClass('ui-tabs-active');
                    
                    if (target === '#lpg-tabs-1' && !$('#lpg-shortcode-builder').data('editing')) {
                        resetForm();
                    }
                });
                
                $('#lpg-tabs-1').show();
                $('.lpg-tab-nav li:first').addClass('ui-tabs-active');
            }
        } catch (error) {
            console.error('Error initializing tabs:', error);
        }
    }
    
    initializeTabs();
    
    // Safe function to get color picker value with fallback
    function getColorPickerValue(elementId, defaultValue) {
        try {
            var element = $(elementId);
            if (element.length && typeof $.fn.wpColorPicker === 'function') {
                var color = element.wpColorPicker('color');
                return color || defaultValue;
            }
            return element.val() || defaultValue;
        } catch (error) {
            console.warn('Error getting color picker value for ' + elementId + ':', error);
            return defaultValue;
        }
    }
    
    // Function to reset form with better error handling
    function resetForm() {
        try {
            $('#lpg-shortcode-builder')[0].reset();
            $('input[name="categories[]"]').prop('checked', false);
            $('#lpg-shortcode-preview').hide();
            
            // Reset to default values
            $('#lpg-show-update-info').val('1');
            $('#lpg-title-capitalization').val('default');
            $('#lpg-columns').val('3');
            $('#lpg-posts-per-page').val('6');
            $('#lpg-order').val('DESC');
            $('#lpg-orderby').val('date');
            $('#lpg-show-featured-image').val('1');
            $('#lpg-button-color').val('#4299e1');
            $('#lpg-button-hover-color').val('#3182ce');
            $('#lpg-pagination-bg-color').val('#ffffff');
            $('#lpg-pagination-text-color').val('#4a5568');
            $('#lpg-pagination-active-color').val('#4299e1');
            $('#lpg-pagination-hover-color').val('#f7fafc');
            $('#lpg-pagination').val('1');
            $('#lpg-pagination-type').val('both');
            
            // Reset color pickers safely
            if (typeof $.fn.wpColorPicker === 'function') {
                try {
                    $('#lpg-button-color').wpColorPicker('color', '#4299e1');
                    $('#lpg-button-hover-color').wpColorPicker('color', '#3182ce');
                    $('#lpg-pagination-bg-color').wpColorPicker('color', '#ffffff');
                    $('#lpg-pagination-text-color').wpColorPicker('color', '#4a5568');
                    $('#lpg-pagination-active-color').wpColorPicker('color', '#4299e1');
                    $('#lpg-pagination-hover-color').wpColorPicker('color', '#f7fafc');
                } catch (error) {
                    console.warn('Error resetting color pickers:', error);
                }
            }
            
            $('button[type="submit"]').text('Save Grid');
            $('#lpg-pagination').trigger('change');
            $('#lpg-title-capitalization').trigger('change');
            $('#lpg-shortcode-builder').removeData('editing');
            
            console.log('Form reset successfully');
        } catch (error) {
            console.error('Error resetting form:', error);
        }
    }
    
    // Toggle pagination type based on pagination setting
    $('#lpg-pagination').on('change', function() {
        if ($(this).val() === '1') {
            $('#lpg-pagination-type').closest('.lpg-column').show();
        } else {
            $('#lpg-pagination-type').closest('.lpg-column').hide();
        }
    });
    
    $('#lpg-pagination').trigger('change');
    
    // Handle shortcode form submission with improved error handling
    $('#lpg-shortcode-builder').on('submit', function(e) {
        e.preventDefault();
        
        try {
            const formData = new FormData(this);
            const shortcodeName = formData.get('shortcode_name');
            
            if (!shortcodeName || shortcodeName.trim() === '') {
                alert('Please enter a shortcode name.');
                return;
            }
            
            const selectedCategories = [];
            $('input[name="categories[]"]:checked').each(function() {
                selectedCategories.push($(this).val());
            });
            
            if (formData.has('categories[]')) {
                formData.delete('categories[]');
            }
            
            selectedCategories.forEach(category => {
                formData.append('categories[]', category);
            });
            
            // Get color picker values with safe fallbacks
            const buttonColor = getColorPickerValue('#lpg-button-color', '#4299e1');
            const buttonHoverColor = getColorPickerValue('#lpg-button-hover-color', '#3182ce');
            const paginationBgColor = getColorPickerValue('#lpg-pagination-bg-color', '#ffffff');
            const paginationTextColor = getColorPickerValue('#lpg-pagination-text-color', '#4a5568');
            const paginationActiveColor = getColorPickerValue('#lpg-pagination-active-color', '#4299e1');
            const paginationHoverColor = getColorPickerValue('#lpg-pagination-hover-color', '#f7fafc');
            
            // Update form data with color values
            formData.set('button_color', buttonColor);
            formData.set('button_hover_color', buttonHoverColor);
            formData.set('pagination_bg_color', paginationBgColor);
            formData.set('pagination_text_color', paginationTextColor);
            formData.set('pagination_active_color', paginationActiveColor);
            formData.set('pagination_hover_color', paginationHoverColor);
            
            // Add action and nonce
            formData.append('action', 'lpg_save_shortcode');
            formData.append('nonce', lpg_admin.nonce);
            
            // Add ID if editing
            if ($(this).data('editing')) {
                formData.append('id', $(this).data('editing'));
            }
            
            // Show loading state
            const submitButton = $(this).find('button[type="submit"]');
            const originalText = submitButton.text();
            submitButton.text('Saving...').prop('disabled', true);
            
            // Send AJAX request
            $.ajax({
                url: lpg_admin.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                timeout: 30000, // 30 second timeout
                success: function(response) {
                    try {
                        if (response && response.success) {
                            // Show success message
                            const preview = $('#lpg-shortcode-preview');
                            preview.html('<div class="lpg-admin-notice success"><p>Grid saved successfully!</p><p>Use this shortcode: <code>' + response.data.shortcode + '</code></p></div>').show();
                            
                            // Reset form if not editing
                            if (!$('#lpg-shortcode-builder').data('editing')) {
                                setTimeout(function() {
                                    resetForm();
                                }, 1000);
                            }
                            
                            // Show success message for longer time instead of reloading
                            setTimeout(function() {
                                preview.html('<div class="lpg-admin-notice success"><p>Grid saved successfully! You can view it in the "Saved Grids" tab.</p></div>');
                            }, 3000);
                            
                            // Reset editing state
                            $('#lpg-shortcode-builder').removeData('editing');
                            submitButton.text('Save Grid');
                            
                            console.log('Grid saved successfully');
                        } else {
                            // Show error message
                            const errorMsg = response && response.data ? response.data.message : 'Unknown error occurred';
                            $('#lpg-shortcode-preview').html('<div class="lpg-admin-notice error"><p>Error: ' + errorMsg + '</p></div>').show();
                            console.error('Save error:', errorMsg);
                        }
                    } catch (error) {
                        console.error('Error processing response:', error);
                        $('#lpg-shortcode-preview').html('<div class="lpg-admin-notice error"><p>Error processing response. Please try again.</p></div>').show();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error:', status, error);
                    let errorMessage = 'An error occurred. Please try again.';
                    if (status === 'timeout') {
                        errorMessage = 'Request timed out. Please try again.';
                    } else if (xhr.status === 403) {
                        errorMessage = 'Permission denied. Please refresh the page and try again.';
                    } else if (xhr.status === 500) {
                        errorMessage = 'Server error. Please check the error logs.';
                    }
                    $('#lpg-shortcode-preview').html('<div class="lpg-admin-notice error"><p>' + errorMessage + '</p></div>').show();
                },
                complete: function() {
                    // Reset button state
                    submitButton.text(originalText).prop('disabled', false);
                }
            });
            
        } catch (error) {
            console.error('Form submission error:', error);
            $('#lpg-shortcode-preview').html('<div class="lpg-admin-notice error"><p>Form submission error. Please refresh the page and try again.</p></div>').show();
            $(this).find('button[type="submit"]').prop('disabled', false);
        }
    });
    
    // Handle copy shortcode button
    $(document).on('click', '.lpg-copy-shortcode', function() {
        const shortcode = $(this).data('shortcode');
        const button = $(this);
        
        // Modern clipboard API
        if (navigator.clipboard && window.isSecureContext) {
            navigator.clipboard.writeText(shortcode).then(function() {
                showCopySuccess(button);
            }).catch(function() {
                fallbackCopyToClipboard(shortcode, button);
            });
        } else {
            fallbackCopyToClipboard(shortcode, button);
        }
    });
    
    // Function to show copy success
    function showCopySuccess(button) {
        const originalHtml = button.html();
        button.html('<span class="dashicons dashicons-yes"></span>');
        
        setTimeout(function() {
            button.html(originalHtml);
        }, 1500);
    }
    
    // Fallback copy function
    function fallbackCopyToClipboard(text, button) {
        const tempInput = $('<input>');
        $('body').append(tempInput);
        tempInput.val(text).select();
        
        try {
            document.execCommand('copy');
            showCopySuccess(button);
        } catch (err) {
            alert('Shortcode: ' + text);
        }
        
        tempInput.remove();
    }
    
    // Handle edit shortcode button with improved error handling
    $(document).on('click', '.lpg-edit-shortcode', function() {
        const id = $(this).data('id');
        
        // Show loading state
        $(this).text('Loading...').prop('disabled', true);
        
        // Load shortcode data
        $.ajax({
            url: lpg_admin.ajax_url,
            type: 'POST',
            data: {
                action: 'lpg_load_shortcode',
                nonce: lpg_admin.nonce,
                id: id
            },
            timeout: 15000, // 15 second timeout
            success: function(response) {
                try {
                    if (response && response.success) {
                        // Switch to create tab
                        if (typeof $.fn.tabs === 'function') {
                            $('#lpg-admin-tabs').tabs('option', 'active', 0);
                        } else {
                            $('.lpg-tab-content').hide();
                            $('.lpg-tab-nav li').removeClass('ui-tabs-active');
                            $('#lpg-tabs-1').show();
                            $('.lpg-tab-nav li:first').addClass('ui-tabs-active');
                        }
                        
                        // Fill form with shortcode data
                        const data = response.data;
                        $('#lpg-shortcode-name').val(data.name || '');
                        $('#lpg-columns').val(data.columns || '3');
                        $('#lpg-posts-per-page').val(data.posts_per_page || '6');
                        $('#lpg-order').val(data.order || 'DESC');
                        $('#lpg-orderby').val(data.orderby || 'date');
                        $('#lpg-show-update-info').val(typeof data.show_update_info !== 'undefined' ? data.show_update_info : '1');
                        $('#lpg-title-capitalization').val(typeof data.title_capitalization !== 'undefined' ? data.title_capitalization : 'default');
                        $('#lpg-show-featured-image').val(typeof data.show_featured_image !== 'undefined' ? data.show_featured_image : '1');
                        
                        // Set button colors safely
                        const buttonColor = typeof data.button_color !== 'undefined' ? data.button_color : '#4299e1';
                        const buttonHoverColor = typeof data.button_hover_color !== 'undefined' ? data.button_hover_color : '#3182ce';
                        
                        $('#lpg-button-color').val(buttonColor);
                        $('#lpg-button-hover-color').val(buttonHoverColor);
                        
                        if (typeof $.fn.wpColorPicker === 'function') {
                            try {
                                $('#lpg-button-color').wpColorPicker('color', buttonColor);
                                $('#lpg-button-hover-color').wpColorPicker('color', buttonHoverColor);
                            } catch (error) {
                                console.warn('Error setting color picker values:', error);
                            }
                        }
                        
                        // Set pagination colors safely
                        const paginationBgColor = typeof data.pagination_bg_color !== 'undefined' ? data.pagination_bg_color : '#ffffff';
                        const paginationTextColor = typeof data.pagination_text_color !== 'undefined' ? data.pagination_text_color : '#4a5568';
                        const paginationActiveColor = typeof data.pagination_active_color !== 'undefined' ? data.pagination_active_color : '#4299e1';
                        const paginationHoverColor = typeof data.pagination_hover_color !== 'undefined' ? data.pagination_hover_color : '#f7fafc';
                        
                        $('#lpg-pagination-bg-color').val(paginationBgColor);
                        $('#lpg-pagination-text-color').val(paginationTextColor);
                        $('#lpg-pagination-active-color').val(paginationActiveColor);
                        $('#lpg-pagination-hover-color').val(paginationHoverColor);
                        
                        if (typeof $.fn.wpColorPicker === 'function') {
                            try {
                                $('#lpg-pagination-bg-color').wpColorPicker('color', paginationBgColor);
                                $('#lpg-pagination-text-color').wpColorPicker('color', paginationTextColor);
                                $('#lpg-pagination-active-color').wpColorPicker('color', paginationActiveColor);
                                $('#lpg-pagination-hover-color').wpColorPicker('color', paginationHoverColor);
                            } catch (error) {
                                console.warn('Error setting pagination color picker values:', error);
                            }
                        }
                        
                        // Set pagination options
                        $('#lpg-pagination').val(typeof data.pagination !== 'undefined' ? data.pagination : '1');
                        $('#lpg-pagination-type').val(typeof data.pagination_type !== 'undefined' ? data.pagination_type : 'both');
                        $('#lpg-pagination').trigger('change');
                        
                        // Reset categories
                        $('input[name="categories[]"]').prop('checked', false);
                        
                        // Check selected categories
                        if (data.categories && data.categories.length) {
                            data.categories.forEach(function(catId) {
                                $('input[name="categories[]"][value="' + catId + '"]').prop('checked', true);
                            });
                        }
                        
                        // Set editing state
                        $('#lpg-shortcode-builder').data('editing', id);
                        $('button[type="submit"]').text('Update Grid');
                        
                        // Show message
                        $('#lpg-shortcode-preview').html('<div class="lpg-admin-notice"><p>Editing grid: ' + (data.name || 'Untitled') + '</p></div>').show();
                        
                        // Scroll to top of form
                        $('html, body').animate({
                            scrollTop: $('#lpg-shortcode-builder').offset().top - 50
                        }, 500);
                        
                        $('#lpg-title-capitalization').trigger('change');
                        
                        console.log('Grid loaded for editing successfully');
                    } else {
                        const errorMsg = response && response.data ? response.data.message : 'Unknown error';
                        alert('Error loading grid: ' + errorMsg);
                    }
                } catch (error) {
                    console.error('Error processing edit response:', error);
                    alert('Error processing response. Please try again.');
                }
            },
            error: function(xhr, status, error) {
                console.error('Edit AJAX error:', status, error);
                alert('An error occurred loading the grid. Please try again.');
            },
            complete: function() {
                // Reset button state
                $('.lpg-edit-shortcode[data-id="' + id + '"]').text('Edit').prop('disabled', false);
            }
        });
    });
    
    // Handle delete shortcode button
    $(document).on('click', '.lpg-delete-shortcode', function() {
        if (!confirm('Are you sure you want to delete this grid? This action cannot be undone.')) {
            return;
        }
        
        const id = $(this).data('id');
        const row = $(this).closest('tr');
        
        // Show loading state
        $(this).text('Deleting...').prop('disabled', true);
        
        // Delete shortcode
        $.ajax({
            url: lpg_admin.ajax_url,
            type: 'POST',
            data: {
                action: 'lpg_delete_shortcode',
                nonce: lpg_admin.nonce,
                id: id
            },
            timeout: 15000,
            success: function(response) {
                try {
                    if (response && response.success) {
                        // Remove row with animation
                        row.fadeOut(300, function() {
                            $(this).remove();
                            
                            // Check if table is empty
                            if ($('.lpg-shortcodes-table tbody tr').length === 0) {
                                $('.lpg-shortcodes-table').replaceWith('<p>No saved grids yet. Create one in the "Create Grid" tab.</p>');
                            }
                        });
                        
                        // Show success message briefly
                        $('<div class="lpg-admin-notice success"><p>' + (response.data.message || 'Grid deleted successfully!') + '</p></div>')
                            .insertBefore('.lpg-shortcodes-table, .lpg-admin-container > p')
                            .delay(3000)
                            .fadeOut();
                            
                        console.log('Grid deleted successfully');
                    } else {
                        const errorMsg = response && response.data ? response.data.message : 'Unknown error';
                        alert('Error deleting grid: ' + errorMsg);
                        $('.lpg-delete-shortcode[data-id="' + id + '"]').text('Delete').prop('disabled', false);
                    }
                } catch (error) {
                    console.error('Error processing delete response:', error);
                    alert('Error processing response. Please try again.');
                    $('.lpg-delete-shortcode[data-id="' + id + '"]').text('Delete').prop('disabled', false);
                }
            },
            error: function(xhr, status, error) {
                console.error('Delete AJAX error:', status, error);
                alert('An error occurred. Please try again.');
                $('.lpg-delete-shortcode[data-id="' + id + '"]').text('Delete').prop('disabled', false);
            }
        });
    });
    
    // Add preview functionality for title capitalization
    $('#lpg-title-capitalization').on('change', function() {
        try {
            const selectedValue = $(this).val();
            const exampleText = 'example post title here';
            let previewText = '';
            
            switch(selectedValue) {
                case 'uppercase':
                    previewText = exampleText.toUpperCase();
                    break;
                case 'lowercase':
                    previewText = exampleText.toLowerCase();
                    break;
                case 'capitalize':
                    previewText = exampleText.replace(/\w\S*/g, function(txt) {
                        return txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase();
                    });
                    break;
                case 'sentence':
                    previewText = exampleText.charAt(0).toUpperCase() + exampleText.slice(1).toLowerCase();
                    break;
                case 'default':
                default:
                    previewText = 'Example Post Title Here';
                    break;
            }
            
            // Show preview
            let previewElement = $('#title-capitalization-preview');
            if (previewElement.length === 0) {
                $(this).after('<div id="title-capitalization-preview"></div>');
                previewElement = $('#title-capitalization-preview');
            }
            
            previewElement.html('Preview: <strong>' + previewText + '</strong>');
        } catch (error) {
            console.error('Error updating title preview:', error);
        }
    });
    
    // Add color preview functionality with error handling
    $('.lpg-color-picker').on('change', function() {
        try {
            updateButtonPreview();
            updatePaginationPreview();
        } catch (error) {
            console.error('Error updating color preview:', error);
        }
    });
    
    function updateButtonPreview() {
        try {
            const buttonColor = getColorPickerValue('#lpg-button-color', '#4299e1');
            const buttonHoverColor = getColorPickerValue('#lpg-button-hover-color', '#3182ce');
            
            // Create or update button preview
            let previewElement = $('#button-color-preview');
            if (previewElement.length === 0) {
                $('#lpg-button-hover-color').closest('.lpg-row-columns').after('<div id="button-color-preview" class="lpg-button-preview"></div>');
                previewElement = $('#button-color-preview');
            }
            
            previewElement.html(
                '<div class="lpg-preview-label">Button Preview:</div>' +
                '<div class="lpg-preview-button" style="background-color: ' + buttonColor + ';">Read More</div>' +
                '<div class="lpg-preview-hover" style="background-color: ' + buttonHoverColor + ';">Read More (Hover)</div>'
            );
        } catch (error) {
            console.error('Error updating button preview:', error);
        }
    }
    
    function updatePaginationPreview() {
        try {
            const paginationBg = getColorPickerValue('#lpg-pagination-bg-color', '#ffffff');
            const paginationText = getColorPickerValue('#lpg-pagination-text-color', '#4a5568');
            const paginationActive = getColorPickerValue('#lpg-pagination-active-color', '#4299e1');
            const paginationHover = getColorPickerValue('#lpg-pagination-hover-color', '#f7fafc');
            
            // Create or update pagination preview
            let previewElement = $('#pagination-color-preview');
            if (previewElement.length === 0) {
                $('#lpg-pagination-hover-color').closest('.lpg-row-columns').after('<div id="pagination-color-preview" class="lpg-pagination-preview"></div>');
                previewElement = $('#pagination-color-preview');
            }
            
            previewElement.html(
                '<div class="lpg-preview-label">Pagination Preview:</div>' +
                '<div class="lpg-pagination-demo">' +
                '<span class="lpg-page-demo" style="background-color: ' + paginationBg + '; color: ' + paginationText + ';">1</span>' +
                '<span class="lpg-page-demo" style="background-color: ' + paginationBg + '; color: ' + paginationText + ';">2</span>' +
                '<span class="lpg-page-demo lpg-active" style="background-color: ' + paginationActive + '; color: #fff;">3</span>' +
                '<span class="lpg-page-demo lpg-hover" style="background-color: ' + paginationHover + '; color: ' + paginationText + ';">4 (Hover)</span>' +
                '</div>'
            );
        } catch (error) {
            console.error('Error updating pagination preview:', error);
        }
    }
    
    // Trigger initial preview
    setTimeout(function() {
        try {
            $('#lpg-title-capitalization').trigger('change');
            updateButtonPreview();
            updatePaginationPreview();
        } catch (error) {
            console.error('Error initializing previews:', error);
        }
    }, 1000);
    
    // Debug information
    console.log('Latest Posts Grid Admin JS Loaded Successfully');
    console.log('jQuery version:', $.fn.jquery);
    console.log('jQuery UI Tabs available:', typeof $.fn.tabs === 'function');
    console.log('WP Color Picker available:', typeof $.fn.wpColorPicker === 'function');
    console.log('AJAX URL:', lpg_admin ? lpg_admin.ajax_url : 'Not available');
});